<?php
/**
 * Plugin Name: Anna Easy SEO Plus
 * Description: A tool to support SEO optimization for search engines that shows how well your texts are optimized. Option to add Sitemap or Snippet preview. SEO scoring of blog posts on a scale of 0–100 points.
 * Version: 1.1.5
 * Author: Anna Nicole Kucerova --  See also: <a href="https://nicoledigital.cz" target="_blank">NicoleDigital Network</a> - Improve your website's performance. See also: <a href="https://www.nicoledigital.cz/novyprojekt/" target="_blank">Eshop struggle help</a>
 */

// Add SEOPlus to the main WordPress menu
add_action('admin_menu', 'asp_add_admin_menu');
add_action('admin_init', 'asp_settings_init');

function asp_add_admin_menu() {
    add_menu_page('SEO Plus', 'EasySEOPlus admin', 'manage_options', 'anna_seo_plus', 'asp_options_page', 'dashicons-chart-line');
}

function asp_settings_init() {
    register_setting('asp_settings_group', 'asp_settings');

    add_settings_section('asp_section', __('SEO Plus Settings', 'wordpress'), null, 'anna_seo_plus');

    add_settings_field('asp_enabled', 'Enable SEO scoring plugin activation', 'asp_enabled_render', 'anna_seo_plus', 'asp_section');
    add_settings_field('asp_h1_title', 'Enter default H1 title for your website', 'asp_h1_title_render', 'anna_seo_plus', 'asp_section');
    add_settings_field('asp_meta_description', 'Default meta description for your website', 'asp_meta_description_render', 'anna_seo_plus', 'asp_section');
}

function asp_enabled_render() {
    $options = get_option('asp_settings') ?? [];
    echo '<input type="checkbox" name="asp_settings[asp_enabled]" value="1"' . checked(1, $options['asp_enabled'] ?? 0, false) . ' />';
}

function asp_h1_title_render() {
    $options = get_option('asp_settings') ?? [];
    echo '<input type="text" name="asp_settings[asp_h1_title]" value="' . esc_attr($options['asp_h1_title'] ?? '') . '" size="50">';
}

function asp_meta_description_render() {
    $options = get_option('asp_settings') ?? [];
    echo '<textarea name="asp_settings[asp_meta_description]" rows="3" cols="50">' . esc_textarea($options['asp_meta_description'] ?? '') . '</textarea>';
}

function asp_options_page() {
    echo '<form action="options.php" method="post">';
    settings_fields('asp_settings_group');
    wp_nonce_field('asp_settings_group', 'asp_nonce');
    do_settings_sections('anna_seo_plus');
    submit_button();
    echo '</form>';
}

// Přidání SEO skóre do seznamu článků
add_filter('manage_edit-post_columns', 'asp_add_seo_column');
add_action('manage_posts_custom_column', 'asp_display_seo_column', 10, 2);

function asp_add_seo_column($columns) {
    $columns['asp_seo_score'] = 'SEOPLUS Score';
    return $columns;
}

function asp_display_seo_column($column, $post_id) {
    if ($column === 'asp_seo_score') {
        $score = asp_calculate_seo_score(get_post($post_id));
        $color = asp_get_score_color($score);
        echo "<strong style='color: $color;'>$score / 100</strong>";
    }
}

// Přidání SEO metaboxu do editoru článku
add_action('add_meta_boxes', 'asp_add_seo_metabox');
function asp_add_seo_metabox() {
    $options = get_option('asp_settings') ?? [];
    if (!empty($options['asp_enabled'])) {
        add_meta_box('asp_seo_score', 'SEO Hodnocení (Anna SEO Plus)', 'asp_render_seo_score', 'post', 'side');
    }
}

function asp_render_seo_score($post) {
    $score = asp_calculate_seo_score($post);
    $color = asp_get_score_color($score);
    
    echo "<p><strong>Skóre: <span style='color:$color;font-size:18px;'>$score / 100</span></strong></p>";
}

// Výpočet skóre SEO na základě obsahu článku
function asp_calculate_seo_score($post) {
    $content = $post->post_content;
    $title = $post->post_title;
    $score = 0;

    // Délka nadpisu
    if (strlen($title) >= 69 && strlen($title) <= 90) {
        $score += 15;
    }

    // Délka článku
    $word_count = str_word_count(strip_tags($content));
    $score += min(floor($word_count / 100) * 10, 20);

    // Zvýrazněná slova
    if (preg_match_all('/<strong>.*?<\/strong>/i', $content, $matches)) {
        $score += min(count($matches[0]) * 5, 10);
    }

    // Obrázek
    if (preg_match('/<img[^>]+>/i', $content)) {
        $score += 5;
    }

    // H2 podnadpis
    if (preg_match('/<h2[^>]*>(.*?)<\/h2>/i', $content, $h2)) {
        if (strlen(strip_tags($h2[1])) >= 60 && strlen(strip_tags($h2[1])) <= 120) {
            $score += 5;
        }
    }

    // Interní a externí odkazy
    if (preg_match_all('/<a[^>]+href=["\'](.*?)["\']/i', $content, $links)) {
        foreach ($links[1] as $url) {
            if (strpos($url, home_url()) !== false) {
                $score += 5;
            } else {
                $score += 2;
            }
        }
    }

    return min($score, 100);
}

// Barvy skóre podle rozsahu
function asp_get_score_color($score) {
    if ($score <= 25) return 'red';
    if ($score <= 40) return 'darkorange';
}

// Přidání informační stránky do nastavení pluginu
add_action('admin_menu', 'asp_add_info_page');

function asp_add_info_page() {
    add_submenu_page(
        'anna_seo_plus',
        'Jak funguje SEO hodnocení',
        'SEO Nápověda',
        'manage_options',
        'asp_seo_help',
        'asp_render_info_page'
    );
}

function asp_render_info_page() {
    ?>
    <div class="wrap">
        <h1>How does SEO scoring work?</h1>
        <p>This plugin evaluates blog posts based on several SEO factors on a scale from <strong>0 to 100 points</strong>. The higher the score, the better optimized the post is for search engines.</p>
        
        <h2>Scoring Factors</h2>
        <ul>
            <li><strong>Title length:</strong> The optimal range is 69–90 characters. You get 15 points.</li>
            <li><strong>Headings H1–H4:</strong> The post should include well-structured headings. You get 5 points for each heading.</li>
            <li><strong>Text length:</strong> You get 5 points for every 100 words. Articles longer than 2000 words receive the maximum score.</li>
            <li><strong>Highlighted words:</strong> Using <code>&lt;strong&gt;</code> tags for important phrases earns 2 points per highlighted word.</li>
            <li><strong>Images:</strong> Including at least one image gives you 5 points.</li>
            <li><strong>H2 subheading:</strong> Ideal length is 60–120 characters. If within range, you get 5 points.</li>
            <li><strong>Internal and external links:</strong> Internal links (to other pages on your site) add 2 points; external links (to other domains) add 1 point.</li>
        </ul>

        <h2>How to improve your post's SEO?</h2>
        <ul>
            <li><strong>Use structured headings:</strong> Break content into logical sections using H1–H4 tags for better readability.</li>
            <li><strong>Write longer articles:</strong> Search engines favor content with at least 300 words.</li>
            <li><strong>Optimize your titles:</strong> Short, clear, and descriptive titles are key to good SEO.</li>
            <li><strong>Add images:</strong> They enhance visual appeal and improve SEO score.</li>
            <li><strong>Include internal links:</strong> Link to other relevant articles on your site.</li>
            <li><strong>Use keywords:</strong> Naturally incorporate keywords in headings and body text.</li>
        </ul>
    </div>
    <?php
}

// Přidání informační stránky "How to use plugin"
add_action('admin_menu', 'asp_add_usage_page');

function asp_add_usage_page() {
    add_submenu_page(
        'anna_seo_plus',
        'Jak používat plugin',
        'Jak používat plugin',
        'manage_options',
        'asp_plugin_usage',
        'asp_render_usage_page'
    );
}

function asp_render_usage_page() {
    ?>
    <div class="wrap">
        <h1>How to Use the Anna SEO Plus Plugin</h1>
        <p><strong>Anna SEO Plus</strong> is a plugin for evaluating the SEO optimization of posts and pages. The plugin allows you to:</p>
        
        <h2>Main Features:</h2>
        <ul>
            <li>✅ SEO scoring for posts in the <strong>Blog Post</strong> and <strong>Pages</strong> sections.</li>
            <li>✅ Add an SEO title and Meta Description directly when editing a post or page.</li>
            <li>✅ Automatically generate a recommended Meta Description (first 160 characters of the content).</li>
            <li>✅ <strong>Google Snippet Preview</strong> – shows how your post will appear in search results.</li>
            <li>✅ Option to set keywords for search engines.</li>
        </ul>

        <h2>How Does the Plugin Work?</h2>
        <p>After installing the plugin, a <strong>SEOPlus</strong> section is added to the WordPress admin menu, where you can enable or disable the plugin.</p>
        
        <h3>How to Optimize a Post Using the Plugin?</h3>
        <ol>
            <li>Open the post or page editor.</li>
            <li>A section called <strong>SEO Optimization - Title, Meta Description & Keywords</strong> will appear below the content.</li>
            <li>Fill in the <strong>SEO Title</strong> (recommended length: 60–90 characters).</li>
            <li>Fill in the <strong>Meta Description</strong> (recommended length: 90–160 characters; 160 is ideal).</li>
            <li>Below these fields, you’ll see a <strong>Google Snippet Preview</strong> – a preview of search engine results.</li>
            <li>You can also add <strong>Keywords for Search Engines</strong>, separated by commas.</li>
            <li>Save the post or page.</li>
        </ol>

        <h3>SEO Scoring:</h3>
        <p>The plugin automatically scores your post on a scale from 0 to 100 points:</p>
        <ul>
            <li><strong>✅ 60–90 characters in the title:</strong> +15 points</li>
            <li><strong>✅ Properly structured H1–H4 headings:</strong> +5 points per heading</li>
            <li><strong>✅ Article length:</strong> +5 points per 100 words</li>
            <li><strong>✅ Highlighted words (<code>&lt;strong&gt;</code>):</strong> +2 points per highlighted word</li>
            <li><strong>✅ At least one image in the post:</strong> +5 points</li>
            <li><strong>✅ H2 subheading of 60–120 characters:</strong> +5 points</li>
            <li><strong>✅ Internal links:</strong> +2 points, External links: +1 point</li>
        </ul>

        <h3>Scoring Colors:</h3>
        <ul>
            <li><span style="color: red;"><strong>0–25 points:</strong> Red (poor)</span></li>
            <li><span style="color: darkorange;"><strong>26–40 points:</strong> Dark orange (needs improvement)</span></li>
            <li><span style="color: goldenrod;"><strong>41–65 points:</strong> Yellow (average)</span></li>
            <li><span style="color: green;"><strong>65–89 points:</strong> Green (good optimization)</span></li>
            <li><span style="color: gold;"><strong>90–100 points:</strong> Gold (excellent SEO)</span></li>
        </ul>

        <p>By optimizing your title, meta description, keywords, and content, you can achieve a higher SEO score!</p>
    </div>
    <?php
}


// Přidání metaboxu pro SEO optimalizaci článků i statických stránek
add_action('add_meta_boxes', 'asp_add_seo_meta_box');
add_action('save_post', 'asp_save_seo_meta_box_data');

function asp_add_seo_meta_box() {
    $post_types = ['post', 'page']; // Zahrnutí článků i statických stránek
    foreach ($post_types as $post_type) {
        add_meta_box(
            'asp_seo_meta',
            'SEO Optimalizace - Nadpis, Meta Popis & Klíčová slova',
            'asp_render_seo_meta_box',
            $post_type,
            'normal'
        );
    }
}

function asp_render_seo_meta_box($post) {
    $seo_title = get_post_meta($post->ID, '_asp_seo_title', true) ?? $post->post_title; // Automatické vyplnění názvu
    $meta_description = get_post_meta($post->ID, '_asp_meta_description', true) ?? substr(strip_tags($post->post_content), 0, 160); // Automatické vyplnění popisu
    $keywords = get_post_meta($post->ID, '_asp_keywords', true) ?? '';
    $post_url = get_permalink($post->ID);

    echo '<div style="display: flex; gap: 20px;">';

    // Levá sekce - formulář
    echo '<div style="width: 60%;">';
    echo '<label for="asp_seo_title"><strong>SEO heading:</strong></label>';
    echo '<input type="text" id="asp_seo_title" name="asp_seo_title" value="' . esc_attr($seo_title) . '" size="80" onkeyup="updateSnippetPreview()" />';
    
    // Hodnocení délky nadpisu
    $title_length = strlen($seo_title);
    $title_score = ($title_length >= 60 && $title_length <= 90) ? '✅ Optimal' : '⚠️ try better';
    echo "<p>Délka nadpisu: <strong id='title-length'>$title_length znaků</strong> - <span id='title-score'>$title_score</span></p>";

    echo '<label for="asp_meta_description"><strong>Meta Popis:</strong></label>';
    echo '<textarea id="asp_meta_description" name="asp_meta_description" rows="3" cols="80" onkeyup="updateSnippetPreview()">' . esc_textarea($meta_description) . '</textarea>';
    
    // Hodnocení délky meta popisu
    $desc_length = strlen($meta_description);
    $desc_score = ($desc_length >= 90 && $desc_length <= 160) ? '✅ Optimal' : '⚠️ try better';
    $quality = ($desc_length == 160) ? '🌟 100% kvalita' : '';
    echo "<p>Délka meta popisu: <strong id='desc-length'>$desc_length znaků</strong> - <span id='desc-score'>$desc_score</span> <span id='desc-quality'>$quality</span></p>";

    echo '<label for="asp_keywords"><strong>Key words for search engine:</strong></label>';
    echo '<input type="text" id="asp_keywords" name="asp_keywords" value="' . esc_attr($keywords) . '" size="80" placeholder="Oddělte klíčová slova čárkou" />';
    echo '<p> Add key words to have better search position (např. „SEO, sport, golf, Google“).</p>';

    echo '</div>';

    // Pravá sekce - Náhled Google Snippet
    echo '<div style="width: 40%; padding: 10px; background: #f8f8f8; border: 1px solid #ddd;">';
    echo '<h4>This is how user will se your snippet in Google search.</h4>';
    echo '<p style="color: #006621; font-size: 14px;" id="snippet-url">' . esc_url($post_url) . '</p>';
    echo '<p style="color: #1a0dab; font-size: 16px; font-weight: bold;" id="snippet-title">' . esc_html($seo_title) . '</p>';
    echo '<p style="color: #4d5156; font-size: 14px;" id="snippet-desc">' . esc_html($meta_description) . '</p>';
    echo '</div>';

    echo '</div>';

    // JavaScript pro aktualizaci náhledu v reálném čase
    echo '<script>
        function updateSnippetPreview() {
            var title = document.getElementById("asp_seo_title").value;
            var desc = document.getElementById("asp_meta_description").value;

            document.getElementById("snippet-title").innerText = title;
            document.getElementById("snippet-desc").innerText = desc;

            document.getElementById("title-length").innerText = title.length + " znaků";
            document.getElementById("desc-length").innerText = desc.length + " znaků";

            document.getElementById("title-score").innerText = (title.length >= 60 && title.length <= 90) ? "✅ Optimal" : "⚠️ Needs to be better";
            document.getElementById("desc-score").innerText = (desc.length >= 90 && desc.length <= 160) ? "✅ Optimal" : "⚠️ Needs to be better";
            document.getElementById("desc-quality").innerText = (desc.length == 160) ? "🌟 100% kvalita" : "";
        }
    </script>';
}

function asp_save_seo_meta_box_data($post_id) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;

    if (isset($_POST['asp_seo_title'])) {
        update_post_meta($post_id, '_asp_seo_title', sanitize_text_field($_POST['asp_seo_title']));
    }

    if (isset($_POST['asp_meta_description'])) {
        update_post_meta($post_id, '_asp_meta_description', sanitize_textarea_field($_POST['asp_meta_description']));
    }

    if (isset($_POST['asp_keywords'])) {
        update_post_meta($post_id, '_asp_keywords', sanitize_text_field($_POST['asp_keywords']));
    }
}

// Přidání postranního SEO panelu do editoru článků a stránek
add_action('add_meta_boxes', 'asp_add_seo_sidebar');

function asp_add_seo_sidebar() {
    $post_types = ['post', 'page'];
    foreach ($post_types as $post_type) {
        add_meta_box(
            'asp_seo_sidebar',
            'SEO Faktory článku',
            'asp_render_seo_sidebar',
            $post_type,
            'side'
        );
    }
}

function asp_render_seo_sidebar($post) {
    $content = $post->post_content;
    $title = $post->post_title;
    $word_count = str_word_count(strip_tags($content));

    // SEO optimization factors and their checks
    $seo_factors = [
        "Title length (60–90 characters)" => strlen($title) >= 60 && strlen($title) <= 90,
        "Use of headings H1–H4" => preg_match("/<h[1-4][^>]*>/i", $content),
        "Article length (minimum 300 words)" => $word_count >= 300,
        "Use of <strong> for important phrases" => preg_match("/<strong>.*?<\/strong>/i", $content),
        "Image in the article" => preg_match("/<img[^>]+>/i", $content),
        "H2 subheading 60–120 characters" => preg_match('/<h2[^>]*>(.*?)<\/h2>/i', $content),
        "Internal links (links to your own site)" => preg_match_all('/<a[^>]+href=["\'](.*?)["\']/i', $content, $links) ? count(array_filter($links[1], fn($url) => strpos($url, home_url()) !== false)) : 0,
        "External links (links to other sites)" => preg_match_all('/<a[^>]+href=["\'](.*?)["\']/i', $content, $links) ? count(array_filter($links[1], fn($url) => strpos($url, home_url()) === false)) : 0
    ];

    echo '<div style="padding: 10px;">';
    echo '<h3>SEO Optimization Factors</h3>';
    echo '<ul>';

    // Display factors with color coding
    foreach ($seo_factors as $factor => $exists) {
        $color = $exists ? 'green' : 'red';
        echo "<li style='color: $color;'>$factor</li>";
    }
}


    echo '</ul>';
    echo '<p style="color: black; font-weight: bold;"> Upozornění (Red text does not include, in your post or page. Try to add them to your content.)</p>';
    echo '</div>';
}

function add_sitemap_main_menu() {
    add_menu_page(
        'SEO Sitemap',  // Název stránky
        'EasySEOplus Sitemap',  // Název v menu
        'manage_options', // Oprávnění
        'seo-sitemap-generator', // Slug stránky
        'seo_sitemap_generator_page', // Callback funkce
        'dashicons-admin-site', // Ikona v menu
        25 // Pozice v hlavním menu
    );
}
add_action('admin_menu', 'add_sitemap_main_menu');

function seo_sitemap_generator_page() {
    if (isset($_POST['generate_sitemap'])) {
        generate_sitemap();
        echo "<div class='updated'><p>The sitemap was successfully generated!</p></div>";
    }

    echo '<div class="wrap">
            <h1>Sitemap Generator</h1>
            <form method="post">
                <input type="submit" name="generate_sitemap" value="Generate Sitemap" class="button button-primary">
            </form>
            
            <hr>

            <h2>What is sitemap.xml?</h2>
            <p><strong>Sitemap.xml</strong> is a file that contains a list of pages on your website. It helps search engines like Google better index your content and improve SEO.</p>

            <h3>Why is it important?</h3>
            <ul>
                <li>Helps search engines find and index new pages faster.</li>
                <li>Ensures that even pages without internal links will be crawled.</li>
                <li>Can contain metadata about pages, such as the last modified date.</li>
            </ul>

            <h3>How often should it be updated?</h3>
            <p>It is recommended to update the sitemap.xml whenever major changes are made to the website, such as publishing new posts or restructuring the site.</p>
          </div>';
}


function generate_sitemap() {
    $posts = get_posts(['post_type' => 'post', 'posts_per_page' => -1]);
    $sitemap = "<?xml version='1.0' encoding='UTF-8'?>\n";
    $sitemap .= "<urlset xmlns='http://www.sitemaps.org/schemas/sitemap/0.9'>\n";

    foreach ($posts as $post) {
        $sitemap .= "<url><loc>" . get_permalink($post) . "</loc></url>\n";
    }

    $sitemap .= "</urlset>";
    file_put_contents(ABSPATH . "sitemap.xml", $sitemap);
}
